
/*******************************************************************************
 * USB connection and RX data handler
 ******************************************************************************/

/* Graphical components binding */
const connectUSBButton = document.querySelector('.app-connect-button');
const connectUSBButtonLabel = document.querySelector('.app-connect-button-label');
const connectUSBBaudrate = new mdc.select.MDCSelect(document.querySelector('.app-baudrate-select'));

/* Global variables */
let portOpened = false;
let keepReading = true;
let port;
let reader;

navigator.serial.addEventListener("disconnect", (event) => {
    closePort();
});

async function closePort() {
    keepReading = false;
    try {
        await reader.cancel();
        await port.close();
    }
    catch {}
    portOpened = false;
    connectUSBButtonLabel.innerHTML = 'Connecter';
    connectUSBBaudrate.disabled = false;
}

async function onConnectButtonClick() {
    if (portOpened == false) {
        try {
            port = await navigator.serial.requestPort();
            await port.open({ baudRate: connectUSBBaudrate.value });
            reader = port.readable.getReader();
            connectUSBButtonLabel.innerHTML = 'Déconnecter';
            let rx_buf = [];
            portOpened = true;
            keepReading = true;
            connectUSBBaudrate.disabled = true;
            while (port.readable && keepReading) {
                try {
                  while (true) {
                    const { value, done } = await reader.read();
                    if (done) {
                        reader.releaseLock();
                        break;
                    }
                    if (value) {
                        rx_buf = new Uint8Array([...rx_buf,...value]);
                        let zeroIndex = rx_buf.indexOf(0);
                        if(zeroIndex != -1) {
                            let cobs_data = rx_buf.slice(0, zeroIndex + 1);
                            message = decode(cobs_data).slice(0,-1);
                            decodeMessage(message);
                            rx_buf = rx_buf.slice(zeroIndex + 1);
                        }
                    }
                  }
                } catch (error) {
                  console.error("Error while reading serial port");
                  console.error(error);
                  await closePort();
                }
            }
            reader.releaseLock();
        }
        catch (err) {
            console.error("Error while opening serial port");
            await closePort();
        }
    }
    else {
        await closePort();
    }
}

/*******************************************************************************
 * RX Message decoder
 ******************************************************************************/

function decodeMessage(message) {
    try {
        const protocol = proto.Message.deserializeBinary(message); //.toObject();
        const protocol_obj = protocol.toObject();
        if (protocol.hasConsole()) {
            const str = protocol.getConsole().getStr();
            terminalAddLine(str);
        }
        else if (protocol.hasHts221()) {
            const humidity = protocol.getHts221().getHumidity();
            const temperature = protocol.getHts221().getTemperature();
            hts221Update(humidity, temperature);
        }
        else if (protocol.hasLps22hh()) {
            const pressure = protocol.getLps22hh().getPressure();
            const temperature = protocol.getLps22hh().getTemperature();
            lps22hhUpdate(pressure, temperature);
        }
        else if (protocol.hasIis2mdctr()) {
            const mag_x = protocol.getIis2mdctr().getMagnetometer().getX();
            const mag_y = protocol.getIis2mdctr().getMagnetometer().getY();
            const mag_z = protocol.getIis2mdctr().getMagnetometer().getZ();
            iis2mdctrUpdate(mag_x, mag_y, mag_z);
            compassCompute(mag_x, mag_y);
        }
        else if (protocol.hasIsm330dhx()) {
            const acc = protocol.getIsm330dhx().getAccelerometer().toObject();
            const gyr = protocol.getIsm330dhx().getGyroscope().toObject();
            ism330dhxUpdate(acc.x, acc.y, acc.z, gyr.x, gyr.y, gyr.z);
            madgwickAHRSupdateIMU(gyr.x * Math.PI / (1000*180), gyr.y * Math.PI / (1000*180), gyr.z * Math.PI / (1000*180), acc.x, acc.y, acc.z, 0.02);
            madgwickQuatToEuler();
            rotateBoard(mdg_yaw, mdg_pitch, mdg_roll);
        }
        else if (protocol.hasAltitude()) {
            const altitude = protocol.getAltitude().getAltitude();
            altitudeUpdate(altitude);
        }
    } catch (error) {
        console.error("Error while decoding message: " + error);
    }
}

/*******************************************************************************
 * HTS221
 ******************************************************************************/

/* Graphical components binding */
const hts221HumidityElement = document.getElementById('hts221_humidity');
const hts221TemperatureElement = document.getElementById('hts221_temperature');

function hts221Update(humidity, temperature) {
/**
 * @type {{humidity: number, temperature: number}}
 */
    hts221HumidityElement.innerText = humidity.toFixed(1) + "%";
    hts221TemperatureElement.innerText = temperature.toFixed(1) + "°C";
}

/*******************************************************************************
 * LPS22HH
 ******************************************************************************/

/* Graphical components binding */
const lps22hhPressureElement = document.getElementById('lps22hh_pressure');
const lps22hhTemperatureElement = document.getElementById('lps22hh_temperature');

function lps22hhUpdate(pressure, temperature) {
/**
 * @type {{pressure: number, temperature: number}}
 */
    lps22hhPressureElement.innerText = pressure.toFixed(0) + " mBar";
    lps22hhTemperatureElement.innerText = temperature.toFixed(1) + "°C";
}

/*******************************************************************************
 * ALTITUDE
 ******************************************************************************/

/* Graphical components binding */
const altitudeElement = document.getElementById('altitude');

function altitudeUpdate(altitude) {
/**
 * @type {{altitude: number}}
 */
    altitudeElement.innerText = altitude.toFixed(2) + " m";
}

/*******************************************************************************
 * IIS2MDCTR
 ******************************************************************************/

/* Graphical components binding */
const magXElement = document.getElementById('mag_x');
const magYElement = document.getElementById('mag_y');
const magZElement = document.getElementById('mag_z');

function iis2mdctrUpdate(x, y, z) {
/**
 * @type {{x: number, y: number, z: number}}
 */
    magXElement.innerText = x.toFixed(3);
    magYElement.innerText = y.toFixed(3);
    magZElement.innerText = z.toFixed(3);
}

/*******************************************************************************
 * ISM330DHX
 ******************************************************************************/

/* Graphical components binding */
const accXElement = document.getElementById('acc_x');
const accYElement = document.getElementById('acc_y');
const accZElement = document.getElementById('acc_z');
const gyrXElement = document.getElementById('gyr_x');
const gyrYElement = document.getElementById('gyr_y');
const gyrZElement = document.getElementById('gyr_z');

function ism330dhxUpdate(acc_x, acc_y, acc_z, gyr_x, gyr_y, gyr_z) {
/**
 * @type {{acc_x: number, acc_y: number, acc_z: number, gyr_x: number, gyr_y: number, gyr_z: number}}
 */
    accXElement.innerText = acc_x.toFixed(0);
    accYElement.innerText = acc_y.toFixed(0);
    accZElement.innerText = acc_z.toFixed(0);
    gyrXElement.innerText = gyr_x.toFixed(0);
    gyrYElement.innerText = gyr_y.toFixed(0);
    gyrZElement.innerText = gyr_z.toFixed(0);
}

/*******************************************************************************
 * Compass rendering
 ******************************************************************************/

/* Graphical components binding */
const compassCanvas = document.getElementById('compass');

/* Global variables */
var compass_background = null,
	compass_needle = null,
	compass_ctx = null,
	compass_degrees = 0;

/* Update compass position angle is in degrees */
function compassDraw(angle) {
	compass_ctx.clearRect(0, 0, 200, 200);
	compass_ctx.drawImage(compass_background, 0, 0);
	compass_ctx.save();
	compass_ctx.translate(100, 100);
	compass_ctx.rotate(angle * (Math.PI / 180));
	compass_ctx.drawImage(compass_needle, -100, -100);
	compass_ctx.restore();
}

/* Initialization */
if (compassCanvas.getContext('2d')) {
    compass_ctx = compassCanvas.getContext('2d');
    compass_needle = new Image();
    compass_needle.src = 'assets/needle.png';
    compass_background = new Image();
    compass_background.src = 'assets/compass.png';
    compass_background.onload = function(){setTimeout(compassDraw(0), 100)};
} else {
    alert("Canvas not supported!");
}

/* Heading calculation helper */
function compassCompute(mag_x, mag_y) {
/**
 * @type {{mag_x: number, mag_y: number}}
 */
    var deg = 0;
    //mag_y = - mag_y;
    //mag_x = - mag_x;
    if (mag_x == 0) mag_x == 0.001;
    if (mag_y == 0) mag_y == 0.001;
    deg = Math.atan2(mag_x,mag_y)*180/Math.PI;
    compassDraw(deg);
}

/*******************************************************************************
 * 3D Orientation rendering
 ******************************************************************************/

/* Graphical components binding */
const canvas = document.getElementById("orientation");

/* Global variables */
var engine = null,
    scene = null,
    sceneToRender = null,
    board = null;

var createDefaultEngine = function () { return new BABYLON.Engine(canvas, true, { preserveDrawingBuffer: true, stencil: true, disableWebGL2Support: false }); };

var createScene = function () {
    var color = new BABYLON.Color4(0.2, 0.2, 0.6, 1);
    var scene = new BABYLON.Scene(engine);
    scene.clearColor = new BABYLON.Color3(1, 1, 1);
    var camera = new BABYLON.FreeCamera("camera1", new BABYLON.Vector3(0, 5, -10), scene);
    camera.setTarget(BABYLON.Vector3.Zero());
    camera.attachControl(canvas, true);
    new BABYLON.HemisphericLight("light", new BABYLON.Vector3(0, 0, -1), scene);
    new BABYLON.HemisphericLight("light", new BABYLON.Vector3(0, 1, 0), scene);
    var ground = BABYLON.MeshBuilder.CreateBox("box", {height: 0.2, width:6, depth: 4, faceColors: new Array(6).fill(color)}, scene);//BABYLON.MeshBuilder.CreateGround("ground", { width: 6, height: 6 }, scene);
    var cone1 = BABYLON.MeshBuilder.CreateCylinder("cone", {height: 1, diameterTop: 0.2, diameterBottom: 0.2, faceColors: new Array(6).fill(color)}, scene);
    cone1.rotation = new BABYLON.Vector3(0, 0, 0.5 * Math.PI);
    cone1.translate(new BABYLON.Vector3(1, 0, 0), 3.5, BABYLON.Space.WORLD);
    var cone2 = BABYLON.MeshBuilder.CreateCylinder("cone", {height: 0.5, diameterTop: 0, diameterBottom: 0.5, faceColors: new Array(6).fill(color)}, scene);
    cone2.rotation = new BABYLON.Vector3(0, 0, -0.5 * Math.PI);
    cone2.translate(new BABYLON.Vector3(1, 0, 0), 4.25, BABYLON.Space.WORLD);
    board = BABYLON.Mesh.MergeMeshes([ground, cone1, cone2]);
    board.rotation = new BABYLON.Vector3(- Math.PI / 4, Math.PI/4, 0);
    return scene;
};
    
engineInit = async function () {
    var asyncEngineCreation = async function () {
        try {
            return createDefaultEngine();
        } catch (e) {
            console.log("the available createEngine function failed. Creating the default engine instead");
            return createDefaultEngine();
        }
    }
    window.engine = await asyncEngineCreation();
    if (!engine) throw 'engine should not be null.';
    engineRenderLoop(engine, canvas);
    window.scene = createScene();
};

engineInit().then(() => {
    sceneToRender = scene;
});

var engineRenderLoop = function (engine, canvas) {
    engine.runRenderLoop(function () {
        if (sceneToRender && sceneToRender.activeCamera) {
            sceneToRender.render();
        }
    });
}

function rotateBoard(yaw, pitch, roll) {
    board.rotation = new BABYLON.Vector3(pitch, -yaw, -roll);
}

/*******************************************************************************
 * Terminal printing
 ******************************************************************************/

/* Graphical components binding */
var terminalContent = document.getElementById("term-content");

function terminalAddLine(str) {
    terminalContent.removeChild(terminalContent.firstElementChild);
    var newElement = document.createElement("p");
    var newText = document.createTextNode(str);
    newElement.appendChild(newText);
    terminalContent.appendChild(newElement);
}

console.log("RUN JAVASCRIPT,  RUUUUUUUUNNNNN !!!!!");