/*
 * protocol.c
 *
 *  Created on: Dec 14, 2023
 *      Author: bmassot
 */

#include "main.h"
#include "pb_encode.h"
#include "protocol.pb.h"
#include "cobs.h"

extern UART_HandleTypeDef huart1;

static Message _message = {0};
static uint8_t _buffer[COBS_INPLACE_SAFE_BUFFER_SIZE] = {0};
static void encode_message(void);

/*
 * Send a text string to interface
 * Length of string MUST be shorter than 200 characters
 */

void Protocol_SendText(char * str, size_t len) {
  _message.which_type = Message_console_tag;
  strncpy(_message.type.console.str, str, sizeof(_message.type.console.str)-1);
  encode_message();
}

/*
 * Send HTS221 sensor values
 */
void Protocol_SendHts221(float humidity, float temperature) {
  _message.which_type = Message_hts221_tag;
  _message.type.hts221.humidity = humidity;
  _message.type.hts221.temperature = temperature;
  encode_message();
}

/*
 * Send LPS22HH sensor values
 */
void Protocol_SendLps22hh(float pressure, float temperature) {
  _message.which_type = Message_lps22hh_tag;
  _message.type.lps22hh.pressure = pressure;
  _message.type.lps22hh.temperature = temperature;
  encode_message();
}

/*
 * Send IIS2MDCTR sensor values
 */
void Protocol_SendIi2smdctr(float mag_x, float mag_y, float mag_z) {
  _message.which_type = Message_iis2mdctr_tag;
  _message.type.iis2mdctr.has_magnetometer = true;
  _message.type.iis2mdctr.magnetometer.x = mag_x;
  _message.type.iis2mdctr.magnetometer.y = mag_y;
  _message.type.iis2mdctr.magnetometer.z = mag_z;
  encode_message();
}

/*
 * Send ISM330DHCX sensor values
 */
void Protocol_SendIsm330dhcx(float acc_x, float acc_y, float acc_z, float gyr_x, float gyr_y, float gyr_z) {
  _message.which_type = Message_ism330dhx_tag;
  _message.type.ism330dhx.has_accelerometer = true;
  _message.type.ism330dhx.has_gyroscope = true;
  _message.type.ism330dhx.accelerometer.x = acc_x;
  _message.type.ism330dhx.accelerometer.y = acc_y;
  _message.type.ism330dhx.accelerometer.z = acc_z;
  _message.type.ism330dhx.gyroscope.x = gyr_x;
  _message.type.ism330dhx.gyroscope.y = gyr_y;
  _message.type.ism330dhx.gyroscope.z = gyr_z;
  encode_message();
}

/*
 * Send altitude value
 */
void Protocol_SendAltitude(float altitude) {
  _message.which_type = Message_altitude_tag;
  _message.type.altitude.altitude = altitude;
  encode_message();
}

/******************************************************************
 * PROTOBUF / COBS ENCODING AND SENDING THROUGH UART
 ******************************************************************/

static void encode_message(void) {
  _buffer[0] = COBS_INPLACE_SENTINEL_VALUE;
  pb_ostream_t ostream = pb_ostream_from_buffer(_buffer + 1, COBS_INPLACE_SAFE_BUFFER_SIZE - 2);
  if (!pb_encode(&ostream, Message_fields, &_message)) {
    return;
  }
  _buffer[ostream.bytes_written + 1] = COBS_INPLACE_SENTINEL_VALUE;
  if (cobs_encode_inplace(_buffer, ostream.bytes_written + 2)) {
    return;
  }
  HAL_UART_Transmit(&huart1, _buffer, ostream.bytes_written + 2, 0xFFFF);
}
