/* INCLUDES */
#include <stdio.h>
#include <stdint.h>
#include "main.h"
#include "project.h"
#include "sensors/hts221.h"
#include "sensors/iis2mdc.h"
#include "sensors/lps22hh.h"
#include "sensors/ism330dhcx.h"
#include "protocol/protocol.h"

/* DEFINES */

/* TYPE DEFINITIONS */

/* GLOBAL VARIABLES */

/* Get peripherals handles from main.c */
extern I2C_HandleTypeDef hi2c2;
extern UART_HandleTypeDef huart1;

/* LOCAL FUNCTION PROTOTYPES */

static void sensors_init(void);
static inline int32_t sensor_read_reg(uint16_t DevAddress, uint16_t MemAddress, uint8_t *pData, uint16_t Size);
static inline int32_t sensor_write_reg(uint16_t DevAddress, uint16_t MemAddress, uint8_t *pData, uint16_t Size);
static inline int32_t sensor_get_tick(void);
static inline void sensor_delay(uint32_t Delay);

/* LOCAL VARIABLES */

/* Buffer for UART transmission */
char uart_buffer[256];
int uart_buffer_size;

/* Sensors configuration */
static HTS221_Object_t hts221 = {0};
static IIS2MDC_Object_t iis2mdc = {0};
static LPS22HH_Object_t lps22hh = {0};
static ISM330DHCX_Object_t ism330dhcx = {0};

/* PUBLIC FUNCTION DEFINITIONS */

uint8_t status;
float humidity, temperature, pressure;
IIS2MDC_Axes_t mag_field;
ISM330DHCX_Axes_t acceleration, angular_rate;

void project_main(void)
{

  sensors_init();

  HTS221_HUM_Enable(&hts221);
  HTS221_TEMP_Enable(&hts221);
  IIS2MDC_MAG_Enable(&iis2mdc);
  LPS22HH_PRESS_Enable(&lps22hh);
  LPS22HH_TEMP_Enable(&lps22hh);
  ISM330DHCX_ACC_Enable(&ism330dhcx);
  ISM330DHCX_GYRO_Enable(&ism330dhcx);

  while(1)
  {
    uint32_t tick1 = HAL_GetTick();
    HTS221_Set_One_Shot(&hts221);
    do
    {
      HTS221_Get_One_Shot_Status(&hts221, &status);
    } while (!status);
    HTS221_HUM_GetHumidity(&hts221, &humidity);
    HTS221_TEMP_GetTemperature(&hts221, &temperature);
    Protocol_SendHts221(humidity, temperature);

    LPS22HH_Set_One_Shot(&lps22hh);
    do
    {
      LPS22HH_Get_One_Shot_Status(&lps22hh, &status);
    } while (!status);
    LPS22HH_PRESS_GetPressure(&lps22hh, &pressure);
    LPS22HH_TEMP_GetTemperature(&lps22hh, &temperature);
    Protocol_SendLps22hh(pressure, temperature);

    IIS2MDC_MAG_GetAxes(&iis2mdc, &mag_field);
    Protocol_SendIi2smdctr(mag_field.x, mag_field.y, mag_field.z);

    ISM330DHCX_ACC_GetAxes(&ism330dhcx, &acceleration);
    ISM330DHCX_GYRO_GetAxes(&ism330dhcx, &angular_rate);
    Protocol_SendIsm330dhcx(acceleration.x, acceleration.y, acceleration.z, angular_rate.x, angular_rate.y, angular_rate.z);

    Protocol_SendAltitude(100.456f);

    uart_buffer_size = snprintf(uart_buffer, 256, "Duration %lu\r\n", HAL_GetTick() - tick1);
    Protocol_SendText(uart_buffer, uart_buffer_size);

    HAL_Delay(100);
  }

}

/********************************************************************************************
 * ******************************************************************************************
 *
 * SENSORS INITIALIZATION
 *
 * ******************************************************************************************
 * *****************************************************************************************/

/* LOCAL FUNCTION DEFINITIONS */

static HTS221_IO_t hts221_io = {
    .BusType = HTS221_I2C_BUS,
    .Address = HTS221_I2C_ADDRESS,
    .ReadReg = sensor_read_reg,
    .WriteReg = sensor_write_reg,
    .Delay = sensor_delay,
    .GetTick = sensor_get_tick
};

static IIS2MDC_IO_t iis2mdc_io = {
    .BusType = IIS2MDC_I2C_BUS,
    .Address = IIS2MDC_I2C_ADD,
    .ReadReg = sensor_read_reg,
    .WriteReg = sensor_write_reg,
    .Delay = sensor_delay,
    .GetTick = sensor_get_tick
};

static LPS22HH_IO_t lps22hh_io = {
    .BusType = LPS22HH_I2C_BUS,
    .Address = LPS22HH_I2C_ADD_H,
    .ReadReg = sensor_read_reg,
    .WriteReg = sensor_write_reg,
    .Delay = sensor_delay,
    .GetTick = sensor_get_tick
};

static ISM330DHCX_IO_t ism330dhcx_io = {
    .BusType = ISM330DHCX_I2C_BUS,
    .Address = ISM330DHCX_I2C_ADD_H,
    .ReadReg = sensor_read_reg,
    .WriteReg = sensor_write_reg,
    .Delay = sensor_delay,
    .GetTick = sensor_get_tick
};

static uint8_t id;

static void sensors_init(void)
{
  /* HTS221 initialization */
  HTS221_RegisterBusIO(&hts221, &hts221_io);
  HTS221_Init(&hts221);
  HTS221_ReadID(&hts221, (uint8_t *)&id);
  if (id != HTS221_ID)
  {
    printf("ERROR: HTS221 ID returned 0x%02X\r\n", id);
  }
  else {
    printf("INFO: HTS221\t\tOK\r\n");
  }

  /* IIS2MDC initialization */
  IIS2MDC_RegisterBusIO(&iis2mdc, &iis2mdc_io);
  IIS2MDC_Init(&iis2mdc);
  IIS2MDC_ReadID(&iis2mdc, (uint8_t *)&id);
  if (id != IIS2MDC_ID)
  {
    printf("ERROR: IIS2MDC ID returned 0x%02X\r\n", id);
  }
  else {
    printf("INFO: IIS2MDC\t\tOK\r\n");
  }

  /* LPS22HH initialization */
  LPS22HH_RegisterBusIO(&lps22hh, &lps22hh_io);
  LPS22HH_Init(&lps22hh);
  LPS22HH_ReadID(&lps22hh, (uint8_t *)&id);
  if (id != LPS22HH_ID)
  {
    printf("ERROR: LPS22HH ID returned 0x%02X\r\n", id);
  }
  else {
    printf("INFO: LPS22HH\t\tOK\r\n");
  }

  /* ISM330DHCX initialization */
  ISM330DHCX_RegisterBusIO(&ism330dhcx, &ism330dhcx_io);
  ISM330DHCX_Init(&ism330dhcx);
  ISM330DHCX_ReadID(&ism330dhcx, (uint8_t *)&id);
  if (id != ISM330DHCX_ID)
  {
    printf("ERROR: ISM330DHCX ID returned 0x%02X\r\n", id);
  }
  else {
    printf("INFO: ISM330DHCX\tOK\r\n");
  }
}

static inline int32_t sensor_read_reg(uint16_t DevAddress, uint16_t MemAddress, uint8_t *pData, uint16_t Size)
{
  return HAL_I2C_Mem_Read(&hi2c2, DevAddress, MemAddress, 1, pData, Size, 10000);
}

static inline int32_t sensor_write_reg(uint16_t DevAddress, uint16_t MemAddress, uint8_t *pData, uint16_t Size)
{
  return HAL_I2C_Mem_Write(&hi2c2, DevAddress, MemAddress, 1, pData, Size, 10000);
}

static inline int32_t sensor_get_tick(void)
{
  return (int32_t) HAL_GetTick();
}

static inline void sensor_delay(uint32_t Delay)
{
  return HAL_Delay(Delay);
}
